/**
 * \file: mspin_measurement.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin Logging
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2014 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_measurement.h"
#include "mspin_logging.h"
#include "mspin_typedef.h"
#include <time.h>
#include <sys/time.h>

#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS

#include <stdio.h>

#define MSPIN_FRAME_MEASUREMENT "FRAME MEASUREMENT"
#define MSPIN_MAX_LEN_TIME_FORMAT 16

static bool gMspinFrameLatencyMeasurement = FALSE;
static U32 gMspinFrameIndex = 0;
static U32 gMspinFrameStartTime = 0; //in microseconds
static U32 gMspinLastStepTime = 0; //in mircoseconds
static U32 gMspinDataReceived = 0; //bytes received

static const tMspinVerbosityLevel gMspinVerbosityLevel = eMspinVerbosityInfo;


static inline U32 mspin_measure_getCurrentTime(void)
{
    struct timeval tv;
    (void)gettimeofday (&tv, NULL);
    return (U32)(tv.tv_sec * 1000000) + (U32)(tv.tv_usec);
}

static inline void mspin_measure_formatTime(char* pFormatedTime, size_t len, U32 printTime)
{
    //Ignore C99 "warning : incompatible implicit declaration" (see https://www.cygwin.com/ml/cygwin/2009-04/msg00437.html)
    snprintf(pFormatedTime, len, "%.2d.%.6d", printTime / 1000000, printTime % 1000000);
}

static inline void mspin_measure_updateFrameIndex(void)
{
    gMspinFrameIndex++;
}

static inline void mspin_measure_printMsg(const char* message, U32 timestamp)
{
    char elapsedTime[MSPIN_MAX_LEN_TIME_FORMAT];
    char totalTime[MSPIN_MAX_LEN_TIME_FORMAT];
    mspin_measure_formatTime(elapsedTime, MSPIN_MAX_LEN_TIME_FORMAT,
            timestamp - gMspinLastStepTime);
    mspin_measure_formatTime(totalTime, MSPIN_MAX_LEN_TIME_FORMAT,
            timestamp - gMspinFrameStartTime);

    mspin_log_printLn(gMspinVerbosityLevel,
            "%s %.6d: d=%sus (t=%sus) %s",
            MSPIN_FRAME_MEASUREMENT, gMspinFrameIndex,
            elapsedTime, totalTime, message);

    gMspinLastStepTime = timestamp;
}
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS

void mspin_measure_enableFrameLatency(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    gMspinFrameLatencyMeasurement = TRUE;
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_reset(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    gMspinFrameStartTime = mspin_measure_getCurrentTime();
    gMspinLastStepTime = gMspinFrameStartTime;
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_startInitLayer(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_printMsg("A.) Start initializing LayerManager", mspin_measure_getCurrentTime());
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_layerManagerInitialized(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_printMsg("B.) LayerManager initialized", mspin_measure_getCurrentTime());
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_startNewFrame(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_updateFrameIndex();
        mspin_measure_reset();

        mspin_measure_printMsg("1.) request new frame", gMspinFrameStartTime);
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameRequested(U8 *pBuffer, size_t bufferSize)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        //A frame update request is 18 bytes and Byte 1 is 0x19
        // (this should be enough to identify a frame update request)
        if ((18 == bufferSize) && pBuffer && (0x19 == pBuffer[0]))
        {
            mspin_measure_printMsg("2.) frame requested", mspin_measure_getCurrentTime());
        }
    }
#else
    MSPIN_UNUSED(pBuffer);
    MSPIN_UNUSED(bufferSize);
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameReceiveStart(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_printMsg("3.) frame receive 1st part", mspin_measure_getCurrentTime());
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameReceived(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        static char message[128];
        //Ignore C99 "warning : incompatible implicit declaration" (see https://www.cygwin.com/ml/cygwin/2009-04/msg00437.html)
        snprintf(message, 128, "4.) frame received with %d bytes", gMspinDataReceived); //the received data might contain a received ping as well. This would be 13 bytes in addition
        mspin_measure_printMsg(message, mspin_measure_getCurrentTime());

        gMspinDataReceived = 0;
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameRectCompressed(U8 frameNumber, U32 bufferSize)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        static char message[128];
        snprintf(message, 128, "5.) frame rect %d with size=%d decompressed", frameNumber+1, bufferSize); //Ignore C99 "warning : incompatible implicit declaration" (see https://www.cygwin.com/ml/cygwin/2009-04/msg00437.html)
        mspin_measure_printMsg(message, mspin_measure_getCurrentTime());
    }
#else
    MSPIN_UNUSED(frameNumber);
    MSPIN_UNUSED(bufferSize);
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameReadyToDisplay(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_printMsg("6.) frame ready to display", mspin_measure_getCurrentTime());
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_frameRendered(void)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        mspin_measure_printMsg("7.) frame rendered", mspin_measure_getCurrentTime());
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
}

void mspin_measure_setDataReceived(U8 *pBuffer, size_t bufferLen)
{
#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    if (gMspinFrameLatencyMeasurement)
    {
        // 1st part of frame?
        if ((gMspinDataReceived == 0) && (bufferLen > 100))
        {
            mspin_measure_frameReceiveStart();
        }

        // Ignore ping requests. Ping requests are identified by 0x80 as first byte and
        // they are 13 bytes in total
        if (pBuffer && (0x80 == pBuffer[0]))
        {
            gMspinDataReceived += bufferLen - 13; //subtract the size of a ping packet
        }
        else if (pBuffer)
        {
            gMspinDataReceived += bufferLen;
        }
    }
#endif //#ifdef MSPIN_ENABLE_FRAME_MEASUREMENTS
    MSPIN_UNUSED(pBuffer);
    MSPIN_UNUSED(bufferLen);
}
